import requests
import sys
from config import API_BASE_URL, get_headers

def check_health():
    """
    Check if the API is reachable and the API Key is valid.
    """
    print(f"Checking API at: {API_BASE_URL}")

    # 1. System Health Check (No Auth Required)
    try:
        response = requests.get(f"{API_BASE_URL}/health")
        if response.status_code == 200:
            print("✅ System Health: OK")
            print(f"   Server Version: {response.json().get('version')}")
        else:
            print(f"❌ System Health Check Failed: {response.status_code}")
            print(response.text)
            sys.exit(1)
    except requests.exceptions.ConnectionError:
        print("❌ Could not connect to server. Is GCRIPT running?")
        sys.exit(1)

    # 2. Authentication Check (Requires Valid Key)
    # We use the config options endpoint which requires auth
    try:
        response = requests.get(
            f"{API_BASE_URL}/config/options",
            headers=get_headers()
        )
        
        if response.status_code == 200:
            print("✅ Authentication: OK (API Key is valid)")
            config = response.json()
            defaults = config.get('defaults', {})
            print(f"   Default Provider: {defaults.get('provider')}")
        elif response.status_code == 401:
            print("❌ Authentication Failed: Invalid API Key")
            print("   Please check your API_KEY in config.py")
        else:
            print(f"❌ unexpected error: {response.status_code}")
            print(response.text)

    except Exception as e:
        print(f"❌ Error during auth check: {str(e)}")

if __name__ == "__main__":
    print("--- GCRIPT API Health Check ---")
    check_health()
    print("-------------------------------")
