import requests
import json
import os
from config import API_BASE_URL, get_headers

def submit_job():
    print(f"Submitting Job to: {API_BASE_URL}/jobs")

    # 1. Create a dummy file if it doesn't exist
    filename = "example_doc.docx"
    if not os.path.exists(filename):
        print(f"Creating dummy file: {filename}")
        with open(filename, "w") as f:
            f.write("This is a test document for the GCRIPT API.")
        # Note: In reality this should be a real .docx file, but for the API check
        # the valid extension is often enough for the initial upload, though processing might fail.
        # Ideally we'd use a real binary .docx, but for a simple stand-alone script we'll just warn.
        print("⚠️  Created a text file with .docx extension. Real processing might fail, but upload will succeed.")

    # 2. Prepare the payload
    # 'file' is the multipart file
    # 'config' is a JSON string with processing options
    files = {
        'file': open(filename, 'rb')
    }
    
    options = {
        "document_type": "report",           # report, proposal, academic, etc.
        "word_count_coverage": "standard",   # standard (skip short), improved, full
        "anonymize_numbers": True,           # Protect sensitive numbers
        "protected_strings": [               # List of custom words to anonymize
            "ConfidentialProject",
            "InternalCodeName",
            "John Doe"
        ],
        "domain": "Technology",              # Optional context domain
        "terminology_focus": "Software Engineering" # Optional terminology hint
    }
    
    data = {
        'config': json.dumps(options)
    }

    try:
        response = requests.post(
            f"{API_BASE_URL}/jobs",
            headers=get_headers(),
            files=files,
            data=data
        )

        if response.status_code == 202:
            job_data = response.json()
            print("✅ Job Submitted Successfully!")
            print(f"   Job ID: {job_data['job_id']}")
            print(f"   Status: {job_data['status']}")
            print("\nSave this Job ID for the next steps!")
            return job_data['job_id']
        else:
            print(f"❌ Job Submission Failed: {response.status_code}")
            print(response.text)
            return None

    except Exception as e:
        print(f"❌ Error submitting job: {str(e)}")
        return None

if __name__ == "__main__":
    submit_job()
