import requests
import json
import os
import time
from config import API_BASE_URL, get_headers

def analyze_and_submit():
    print(f"--- Advanced Workflow: Analyze & Skip Sections ---")
    
    # 1. Create a dummy file if it doesn't exist
    filename = "advanced_doc.docx"
    if not os.path.exists(filename):
        with open(filename, "w") as f:
            f.write("Dummy content for analysis.")
        print(f"Created dummy file: {filename}")

    # 2. Analyze Sections
    print(f"\n1. Analyzing document structure...")
    try:
        with open(filename, 'rb') as f:
            files = {'file': f}
            response = requests.post(
                f"{API_BASE_URL}/analyze-sections",
                headers=get_headers(), # Authentication might be needed depending on config
                files=files
            )
        
        if response.status_code != 200:
            print(f"❌ Analysis failed: {response.text}")
            return

        analysis = response.json()
        sections = analysis.get('sections', [])
        print(f"✅ Found {len(sections)} sections:")
        
        # Display sections for user to choose
        for i, section in enumerate(sections):
            # Try to get a title or preview
            title = section.get('title') or "Untitled Section"
            preview = section.get('content_preview', '')[:50]
            print(f"   [{i}] {title} - '{preview}...'")

    except Exception as e:
        print(f"❌ Error during analysis: {e}")
        return

    # 3. Define Sections to Skip (Simulation)
    # in a real app, you would let the user pick.
    # We will skip the second section (index 1) if it exists.
    skip_indices = []
    if len(sections) > 1:
        skip_indices = [1] 
        print(f"\n2. Configuring to SKIP section index: {skip_indices}")
    else:
        print("\n2. Document too short to skip sections.")

    # 4. Submit Job with Skip Config
    print(f"\n3. Submitting Job with skip config...")
    
    options = {
        "document_type": "report",
        "skip_sections": skip_indices,  # <--- MAGIC FIELD (List of Integers)
        "anonymize_numbers": True
    }
    
    try:
        with open(filename, 'rb') as f:
            files = {'file': f}
            data = {'config': json.dumps(options)}
            
            response = requests.post(
                f"{API_BASE_URL}/jobs",
                headers=get_headers(),
                files=files,
                data=data
            )
            
        if response.status_code == 202:
            job_data = response.json()
            print(f"✅ Job Submitted! ID: {job_data['job_id']}")
            print(f"   The job will process only the non-skipped sections.")
        else:
            print(f"❌ Submission failed: {response.text}")

    except Exception as e:
        print(f"❌ Error submitting job: {e}")

if __name__ == "__main__":
    analyze_and_submit()
